; This file holds functions that create and manipulate the cache of header
; information for the various message files. It is explicitly loaded from
; the root.
-------------------------------------------------------------------------

; This function creates the header buffer that represents a message or
; bboard directory. It associates several buffer-specific variables
; with it: mh-buffer-filename, which is the actual Unix file name of the
; directory; mh-folder-title, which is either the tail of the directory
; name or the whole thing depending on whether it is in your mail path.
; call: (&mh-read-folder "folder" "range" "pathname" "title")
(defun 
    (&mh-read-folder name rnge title
	(setq name (arg 1)) (setq rnge (arg 2))
	(&mh-pop-to-buffer (concat "+" mh-folder))
	(if buffer-is-modified
	    (write-current-file))
	(setq mh-folder name)
	(switch-to-buffer (concat "+" name))
	(setq needs-checkpointing 0)
	(setq window-priority 40)
	(setq mode-string "mh-folder")
	(setq mh-buffer-filename (arg 3))
	(setq mh-folder-title (arg 4))
	(use-local-map "&mh-keymap")

	(if (= (buffer-size) 0)
	    (progn 
		   (if (!= 0 (length mh-flist))
		       (setq mh-flist (concat mh-flist ",")))
		   (setq mh-flist (concat mh-flist name))
		   (if (error-occured 
			   (read-file (concat mh-buffer-filename "/"
					      (&mh-header-file-name))))
		       (progn 
			      (message "Header file missing;  regenerating it...")
			      (sit-for 0)
			      (&mh-regenerate-headers)
		       )
		       (&mh-add-inc-headers)
		   )
		   (if (looking-at "No messages ")
		       (progn 
			      (if (= rnge "")
				  (message  "Folder +" name " is empty.")
				  (message  "No messages in +" name " range " rnge)
				  (erase-buffer)
			      )
			      (sit-for 15)
		       )
		   )
		   (setq mode-line-format mh-mode-line)
		   (&mh-check-folder-full)
	    )
	)
	(if (file-exists (concat mh-buffer-filename "/++update"))
	    (progn
		  (end-of-file) (previous-line)
		  (message "Updating message headers, please wait")
		  (sit-for 0)
		  (if (&mh-get-new-headers)
		      (&mh-set-cur)
		  )
	    )
	)
	(&mh-adjust-window)
	(setq buffer-is-modified 0)
    )
)

(defun
    (&mh-check-folder-full lastmsg
	(save-excursion
	    (temp-use-buffer (concat "+" mh-folder))
	    (end-of-file)
	    (previous-line)
	    (beginning-of-line)
	    (while (= (following-char) ' ') (forward-character))
	    (set-mark)
	    (beginning-of-line)
	    (goto-character (+ (dot) mh-msgnum-cols))
	    (setq lastmsg (region-to-string))
	    (if (> lastmsg (if mh-version-6-or-later 
			       9000
			       900))
		(progn ans
		       (setq ans 
			     (get-response (concat "Folder +" mh-folder " is >90%% full. May I pack it for you? ")
				 "yYnN\"
				 "Please answer y or n"))
		       (if (= ans 'y')
			   (progn
				 (&mh-pack-folder)
			   )
			   (progn
				 (message "OK, but you should use the 'x-p' command to pack it soon.")
				 (sit-for 20)
			   )
		       )
		)
	    )
	)
    )
    
    (&mh-adjust-window
	(&mh-unmark-all-headers 0)
	(&mh-position-to-current)
;	(save-excursion 
;	    (beginning-of-window)
;	    (if (! (bobp))
;		(progn t
;		       (end-of-file)
;		       (setq t (dot))
;		       (while (= t (dot))
;			      (progn 
;				     (scroll-one-line-down)
;				     (sit-for 0)
;			      ))
;		       (scroll-one-line-up)
;		)
;	    )
;	)
    )
)

(defun 
    (&mh-regenerate-headers
	(setq mode-line-format " please wait for header regeneration...")
	(message  "scan +" mh-folder-title mh-width)
	(sit-for 0)
	(erase-buffer) (set-mark)
	(fast-filter-region  (concat mh-progs "/scan +" mh-folder-title mh-width))
	(beginning-of-file)
	(error-occured (re-replace-string (if mh-version-6-or-later
					      "^\\(.....\\)-"
					      "^\\(....\\)-")
			   "\\1R"))
	(write-named-file (concat mh-buffer-filename "/"
				  (&mh-header-file-name)))
	(unlink-file (concat mh-buffer-filename "/++"))
	(unlink-file (concat mh-buffer-filename "/++update"))
	(message "...done")
    )
    
    
    (&mh-header-file-name folder-name pos
	(if (!= (substr (current-buffer-name) 2 1) "/")
	    
	    (save-excursion
		(setq folder-name (current-buffer-name))
		(temp-use-buffer "+buf+")
		(erase-buffer)
		(beginning-of-file)
		(insert-string folder-name)
		(beginning-of-file)
		(error-occured
		    (replace-string "/" ".")
		)
		(beginning-of-file)
		(set-mark)
		(end-of-file)
		(region-to-string)
	    )
	    
	    ; else
	    (progn
		  (setq folder-name (current-buffer-name))
		  (setq pos -1)
		  (while (!= (substr folder-name pos 1) "/") (setq pos (- pos 1)))
		  (concat "+" (substr folder-name (+ pos 1) (- 0 pos)))
	    )
	)
    )
)

; Read in the ++ file that was generated by an external "inc", then erase.
(defun 
    (&mh-add-inc-headers uhf
	(setq uhf (concat mh-buffer-filename "/++"))
	(if (file-exists uhf)
	    (progn 
		   (save-excursion
		       (temp-use-buffer "++")
		       (read-file uhf)
		       (temp-use-buffer (concat "+" mh-folder))
		       (end-of-file)
		       (yank-buffer "++")
		       (write-current-file)
		       (temp-use-buffer "++")
		       (erase-buffer)
		       (write-current-file)
		       (unlink-file uhf)
		   )
	    )
	)
    )


; Check to see of any messages after the last one in the folder cache exist.
; If so, then scan from there through "last", appending to the end of the
; cache.  Return 1 if new headers were found, 0 otherwise.
    (&mh-get-new-headers num mk
	(unlink-file (concat mh-buffer-filename "/++update"))
	(temp-use-buffer (concat "+" mh-folder))
	(end-of-file)
	(if (bolp) (previous-line) (beginning-of-line))
	(setq num (&mh-get-msgnum))
	(end-of-file) (set-mark)
	(fast-filter-region
	    (concat mh-progs "/scan +" mh-folder-title " " (+ num 1) "-last" mh-width))
	(exchange-dot-and-mark)
	(beginning-of-line)
	(setq mk (dot))			;  Beginning of result
	(error-occured (re-replace-string (if mh-version-6-or-later
					      "^\\(.....\\)-"
					      "^\\(....\\)-")
			   "\\1R"))
	(error-occured
	    (re-search-forward "^[a-zA-Z]")
	    (beginning-of-line)
	    (set-mark) (end-of-file)
	    (erase-region)
	    (previous-line)
	)
	;  Did we find any new messages?  Return 1 if so.
	(end-of-file)
	(if (> (dot) mk)
	    (progn
		  (write-named-file (concat mh-buffer-filename "/"
					    (&mh-header-file-name)))
		  1
	    )
	    0
	)
    )

    (&mh-scan-filter
	(save-excursion
	    (temp-use-buffer (current-process))
	    (end-of-file)
	    (set-mark)
	    (insert-string (process-output))
	    (exchange-dot-and-mark)
	    (beginning-of-line)
	    (error-occured (re-replace-string (if mh-version-6-or-later
						  "^\\(.....\\)-"
						  "^\\(....\\)-")
			       "\\1R"))
	    (error-occured
		(re-search-forward "^[a-zA-Z]")
		(set-mark)
		(end-of-file)
		(erase-region)
	    )
	    (write-named-file (concat mh-buffer-filename "/"
				      (&mh-header-file-name)))
	)
    )
)



; This function removes all "+" flags from the headers, and if it is called
; with an argument of 1, removes all "D" and "^" flags too.
(defun
    (&mh-unmark-all-headers search-pattern
	(temp-use-buffer (concat "+" mh-folder))
	(setq search-pattern 
	      (if mh-version-6-or-later
		  (if (= 0 (arg 1))
		      "^....\\+"
		      "^....\\D\\|^....\\^\\|^....\\+")
		  (if (= 0 (arg 1))
		      "^...\\+"
		      "^...\\D\\|^...\\^\\|^...\\+")))
	(beginning-of-file)
	(while (! (error-occured (re-search-forward search-pattern)))
	       (delete-previous-character)
	       (insert-character ' ')
	)
    )
    
; position the cursor to the current message.
    (&mh-position-to-current curmsg curfil
	(setq curfil mh-buffer-filename)
	(if mh-version-6-or-later
	    (progn
		  (setq curmsg (&mh-find-key
				   (concat curfil "/.mh_sequences") "cur"))
		  (if (= curmsg "") (setq curmsg 0))
	    )
	    (save-excursion
		(temp-use-buffer "mh-temp") (erase-buffer)
		(if (error-occured
			(insert-file (concat curfil "/cur")))
		    (setq curmsg 0)
		    (progn
			  (beginning-of-file)
			  (set-mark)
			  (end-of-line)
			  (setq curmsg (region-to-string))
		    )
		)
	    )
	)
	(end-of-file)
	(if (= curmsg 0) (previous-line)
	    (progn
		  (if (error-occured 
			  (re-search-reverse (concat "^[ ]*" curmsg)))
		      (progn (end-of-file) (previous-line))
		  )
	    )
	)
	(if (! (eobp))
	    (progn
		  (beginning-of-line)
		  (goto-character (+ (dot) mh-msgnum-cols))
 		  (if (! (looking-at "D\\|\\^"))
 		      (progn (delete-next-character)
 			     (insert-character '+')))
		  (beginning-of-line)
	    )
	)
    )

; This function sets the "current message" (+ sign) to equal the number of
; the message that the cursor is pointing to. I.e. it writes cur to stable
; storage
    (&mh-set-cur curfil cm
	(temp-use-buffer (concat "+" mh-folder))
	(setq curfil mh-buffer-filename)
	(setq cm (&mh-get-msgnum))
	(if mh-version-6-or-later
	    (&mh-find-key (concat curfil "/.mh_sequences") "cur" cm)
	    (progn cf
		   (setq cf (concat curfil "/cur"))
		   (temp-use-buffer "mh-temp")
		   (erase-buffer)
		   (insert-string cm)
		   (insert-string "\n")
		   (write-named-file cf)
		   (delete-buffer "mh-temp")
		   (temp-use-buffer "mh-temp")
		   (setq needs-checkpointing 0)
	    )
	)
    )

; write out the header buffer as a file in the current folder
    (&mh-make-headers-current search-pattern
	(setq search-pattern 
	      (if mh-version-6-or-later
		  "^....\\D\\|^....\\^"
		  "^...\\D\\|^...\\^"))
	(temp-use-buffer (concat "+" mh-folder))
	(save-excursion 
	    (beginning-of-file)
	    (while (! (error-occured
			  (re-search-forward search-pattern)))
		   (beginning-of-line)
		   (kill-to-end-of-line) (delete-next-character)
	    )
	    (write-current-file)
	)
	(&mh-set-cur)
    )

; This function closes a folder, i.e. processes all of the pending deletes and
    ; moves for it and edits the header buffer accordingly.
    (&mh-close-folder ts sscmd
	(temp-use-buffer "cmd-buffer") (beginning-of-file)
	;  Process removes
	(message ".") (sit-for 0)
	(error-occured 
	    (re-search-forward (concat "^rmm +" mh-folder))
	    (beginning-of-line) (insert-string mh-progs "/")
	    (beginning-of-line) (set-mark)
	    (end-of-line) (delete-next-character)
	    (setq ts (region-to-string)) (erase-region)
	    (send-to-shell ts 'f')
	)
	;  Process moves
	(beginning-of-file)
	(setq sscmd (concat "^" mh-file-command " -src +" mh-folder))
	(message "..") (sit-for 0)
	(while (! (error-occured
		      (re-search-forward sscmd)
		  ))
	       ; Mark destination folder.
	       (if (looking-at ".*+\\([^ \t]*\\)")
		   (progn dest
			  (region-around-match 1)
			  (setq dest (concat mh-path "/" (region-to-string)
					     "/++update"))
			  (if (! (file-exists dest))
			      (save-excursion 
				  (temp-use-buffer "++update") (erase-buffer)
				  (insert-string
				      "header cache needs to be updated.\n")
				  (write-named-file dest)
				  (erase-buffer)
			      )
			  )
		   )
	       )
	       (beginning-of-line) (insert-string mh-progs "/")
	       (beginning-of-line) (set-mark)
	       (end-of-line) (delete-next-character)
	       (setq ts (region-to-string)) (erase-region)
	       (send-to-shell ts 'f')
	)
	(message "...") (sit-for 0)
	(&mh-pop-to-buffer (concat "+" mh-folder))
	(&mh-make-headers-current)
	(message "....") (sit-for 0)
	(&mh-unmark-all-headers 1)
	; (message ".....") (sit-for 0)		Caused screen to shift
	(&mh-position-to-current)
	(message ".....") (sit-for 0)
	(write-current-file)
	(message "..... done") (sit-for 0)
    )

;  This function applies "folder -pack" to the current folder, after first
;  closing it (see above)
    (&mh-pack-folder
	(setq mode-line-format " closing folder first...") (sit-for 0)
	(&mh-close-folder)
	(setq mode-line-format " please wait for pack...") (sit-for 0)
	(send-to-shell (concat mh-progs "/folder +" mh-folder " -pack") 'f')
	(&mh-regenerate-headers)
	(setq mode-line-format mh-mode-line)
    )
)
